<?php

namespace Tests\Feature;

use App\User;
use App\Video;
use App\VideoCategoryDescription;
use App\WebsiteCategoryDescription;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;
use Tests\TestCase;

class VideoCategoryTest extends TestCase
{
    // To reset DB to it's status before run test
    use DatabaseTransactions;

    // To create fake sentences to help us test and validation
    use WithFaker;

    private $user;


    protected function setUp() :void
    {
        parent::setUp();

        // to see description
        $this->withoutExceptionHandling();

        // Start session to enables csrf_token()
        Session::start();

        // Authenticate user
        $this->user = factory(User::class)->create(['role'=>'admin']);

        $this->actingAs($this->user);
    }

    /**
     * Method to ensure that the user can access to video category.
     *
     * @return void
     */
    public function testUserCanAccessVideoCategoryPage() :void
    {

        // Video gallery route which visit by user
        $response = $this->get(route('videoCategories.index'));

        // Should be return status 200
        $response->assertStatus(200);
    }

    /**
     * Method to ensure that the user can read all video categories.
     *
     * @return void
     */
    public function testUserCanReadAllVideoCategories() :void
    {
        //Given we have gallery in the database
        $videoCategory = factory(VideoCategoryDescription::class)->create();

        //When user visit the video category
        $response = $this->get(route('videoCategories.grid'));
        // status should be 200
        $response->assertStatus(200);

        //He should be able to read the video category
        $response->assertSee($videoCategory->name);

    }

    /**
     * Method to ensure that the create form route exists.
     *
     * @return void
     */

    public function testUserCanCreateVideoCategory()
    {
        //When user visit the new video category form route
        $response = $this->get(route('videoCategories.create'));

        //He should be able to see the fields which enable him to add new category
        $response->assertStatus(200);
        $name = metaFields('videoCategories', 'name', getCurrentLocale());
        $response->assertSee($name ?? __('videoCategories.name'));

    }

    /**
     * Method to ensure that the update form route exists.
     *
     * @return void
     */

    public function testUserCanEditVideoCategory()
    {
        //Given we have video category in the database
        $videoCategory = factory(VideoCategoryDescription::class)->create();

        //When user visit the video category form route
        $response = $this->get(route('videoCategories.edit', $videoCategory->video_category_id));

        //He should be able to see the fields which enable him to edit the video category
        $response->assertStatus(200);
        $name = metaFields('videoCategories', 'name', getCurrentLocale());
        $response->assertSee($name ?? __('videoCategories.name'));

        $response->assertSee($videoCategory['name_'.getCurrentLocale()]);
    }

    /**
     * Method to ensure that the user can add video category and video category description.
     *
     * @return void
     */
    public function testUserCanAddVideoGallery() :void
    {

        $dataToSave = ['image' => '/uploads/images/5eafffbad512e.jpg'];

        foreach (languages() as $key => $language) {
            $dataToSave['name_'.$language->local] = Str::random(16);
            $dataToSave['slug_'.$language->local] = Str::random(16);
            $dataToSave['keywords_'.$language->local] = Str::random(16);
            $dataToSave['meta_description_'.$language->local] = Str::random(16);
        }

        //When user submits post request to create video category endpoint
        $response= $this->post(route('videoCategories.store'), array_merge( $dataToSave, ['_token'=> csrf_token()]));

        // The redirect response header status is 302
        $response->assertStatus(302);

        // The response redirect to video categories
        $response->assertRedirect('admin/videoCategories');


        //It gets stored in the database
        $this->assertDatabaseHas('video_categories',
            [
                'image'=> $dataToSave['image'],
            ]
        );

        foreach (languages() as $key => $language) {
            $this->assertDatabaseHas('video_category_descriptions',
                [
                    'name' => $dataToSave['name_' . $language->local],
                    'slug' => $dataToSave['slug_' . $language->local],
                    'keywords' => $dataToSave['keywords_' . $language->local],
                    'meta_description' => $dataToSave['meta_description_' . $language->local],
                ]
            );
        }

        // Session success message
        $response->assertSessionHas('message', __('dashboard.saveDone'));

        // Page appears in the Pages panel
        $response = $this->get(route('videoCategories.grid'));

        //He should be able to read the video category
        $response->assertSee($dataToSave['name_'.getCurrentLocale()]);

    }

    /**
     * Method to ensure that the user can Edit video category.
     *
     * @return void
     */
    public function testUserCanUpdateVideoCategory() :void
    {

        //Add a video category object
        $videoCategory = factory(VideoCategoryDescription::class)->make(['language_id' => 0]);

        $dataToSave = ['image' => '/uploads/images/5eafffbad512e.jpg'];

        foreach (languages() as $key => $language) {
            $dataToSave['name_'.$language->local] = Str::random(16);
            $dataToSave['slug_'.$language->local] = Str::random(16);
            $dataToSave['keywords_'.$language->local] = Str::random(16);
            $dataToSave['meta_description_'.$language->local] = Str::random(16);
        }

        //When user submits post request to edit video category endpoint
        $response= $this->put(route('videoCategories.update', $videoCategory->video_category_id),array_merge( $dataToSave, ['_token'=> csrf_token()] ) );

        // The redirect response header status is 302
        $response->assertStatus(302);

        // The response redirect to video categories
        $response->assertRedirect('/admin/videoCategories');;

        //It gets stored in the database
        $this->assertDatabaseHas('video_categories',
            [
                'image'=> $dataToSave['image'],
            ]
        );

        foreach (languages() as $key => $language) {
            $this->assertDatabaseHas('video_category_descriptions',
                [
                    'name'=> $dataToSave['name_'.$language->local],
                    'slug'=> $dataToSave['slug_'.$language->local],
                    'keywords'=> $dataToSave['keywords_'.$language->local],
                    'meta_description'=> $dataToSave['meta_description_'.$language->local],
                    'language_id'=> $language->id,
                    'video_category_id'=> $videoCategory->video_category_id,
                ]
            );
        }


        // Session success message
        $response->assertSessionHas('message', __('dashboard.saveDone'));

        // Page appears in the Page grid
        $response = $this->get(route('videoCategories.grid'));

        //He should be able to read the video category
        $response->assertSee($dataToSave['name_'.getCurrentLocale()]);
    }

    /**
     * Method to ensure that the user can send the video category to trash.
     *
     * @return void
     */
    public function testVideoCategoriesTrash() :void
    {

        //Add a video category object
        $videoCategory = factory(VideoCategoryDescription::class)->make();

        //When the user hit's with endpoint to delete the video category
        $this->delete(route('videoCategories.destroy', $videoCategory->video_category_id), ['_token'=> csrf_token()]);

        //The video category should be deleted from the database.
        $this->assertSoftDeleted('video_categories',['id'=> $videoCategory->video_category_id]);
    }

    /**
     * Method to ensure that the user can delete the video category from database.
     *
     * @return void
     */
    public function testVideoCategoriesDelete() :void
    {
        //Add a video category object
        $videoCategory =  factory(VideoCategoryDescription::class)->make();

        // user sent the video category to trash first, as he can not delete it from the first click
        $this->delete(route('videoCategories.destroy', $videoCategory->video_category_id), ['_token'=> csrf_token()]);

        //When the user hit's the endpoint to delete the video category
        $this->delete(route('videoCategories.destroy', $videoCategory->video_category_id), ['_token'=> csrf_token()]);

        //The video category should be deleted from the database.

        $this->assertDatabaseMissing('video_categories',['id'=> $videoCategory->video_category_id]);

        $this->assertDatabaseMissing('video_category_descriptions',['video_category_id'=> $videoCategory->video_category_id]);
    }

    /**
     * Method to ensure that the user can send multiple video categories to trash.
     *
     * @return void
     */
    public function testVideoCategoriesMultiTrash() :void
    {
        //Add a video category object
        $videoCategory = factory(VideoCategoryDescription::class, 3)->make();

        $ids= $videoCategory->pluck('video_category_id')->toArray();

        //When the user hit's the endpoint to send the video categories to trash
        $this->delete(route('videoCategories.destroyAll'), ['_token'=> csrf_token(), 'ids'=> $ids]);
        //The video category should be deleted from the database.
        $this->assertSoftDeleted('video_categories',['id'=> $ids]);
    }

    /**
     * Method to ensure that the user can delete multiple video categories.
     *
     * @return void
     */
    public function testVideoCategoriesMultiDelete() :void
    {
        //Add a video category object
        $videoCategory = factory(VideoCategoryDescription::class, 3)->make();

        $ids= $videoCategory->pluck('video_category_id')->toArray();

        //When the user hit's the endpoint to send the video categories to trash
        $this->delete(route('videoCategories.destroyAll'), ['_token'=> csrf_token(), 'ids'=> $ids]);

        //When the user hit's the endpoint to delete the video categories from the db
        $this->delete(route('videoCategories.destroyAll'), ['_token'=> csrf_token(), 'ids'=> $ids, 'force' => true]);

        //The video category should be deleted from the database.
        $this->assertDatabaseMissing('video_categories',['id'=> $ids]);

        $this->assertDatabaseMissing('video_category_descriptions',['video_category_id'=> $ids]);

    }

    /**
     * Method to ensure that the user can restore the video category from trash.
     *
     * @return void
     */
    public function testVideoCategoriesRestore() :void
    {
        //Add a video category object
        $videoCategory =  factory(VideoCategoryDescription::class)->make();

        //the user send the video category to trash
        $this->delete(route('videoCategories.destroy', $videoCategory->video_category_id), ['_token'=> csrf_token()]);

        $this->assertSoftDeleted('video_categories',['id'=> $videoCategory->video_category_id]);

        //the user restore the video category
        $this->put(route('videoCategories.restore', $videoCategory->video_category_id), ['_token'=> csrf_token()]);

        //The video category should be restored .
        $this->assertDatabaseHas('video_categories',['id'=> $videoCategory->video_category_id, 'deleted_at'=> null]);
    }

    /**
     * Method to ensure that the user can restore multiple video categories.
     *
     * @return void
     */
    public function testVideoCategoriesMultiRestore() :void
    {
        //Add a video category object
        $videoCategory = factory(VideoCategoryDescription::class, 3)->make();

        $ids= $videoCategory->pluck('video_category_id')->toArray();

        //When the user hit's the endpoint to send the video category to trash
        $response = $this->delete(route('videoCategories.destroyAll'), ['_token' => csrf_token(), 'ids' => $ids]);

        // Test last one to ensure soft deleted process done successfully
        $this->assertSoftDeleted('video_categories', ['id' => $ids]);

        //When the user hit's the endpoint to restore the video categories from the trash
        $this->put(route('videoCategories.restoreAll'), ['_token' => csrf_token(), 'ids' => $ids]);

        // Test last one to ensure restore process done successfully
        $this->assertDatabaseHas('video_categories', ['id' => $ids, 'deleted_at' => null]);

    }
}
