<?php

namespace App;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;

/**
 * Class Page
 * @package App
 * @mixin Builder
 */
class Page extends Model
{
    // Fields that mass assignment
    protected $fillable = ['image', 'images', 'meta'];

    // Use soft delete
    use SoftDeletes;

    protected $casts = ['images' => 'array', 'meta' => 'array'];

    /**
     * Return Slug of page
     * @return string|null
     */
    public function getSlugTextAttribute() : ?string
    {
        return str_slug(strtolower($this->currentDescription->slug));
    }

    /**
     * Get description for pages
     * @return HasMany
     */
    public function descriptions() :HasMany
    {
        return $this->hasMany(PageDescription::class);
    }

    /**
     * Get description of current language for pages
     * @return HasOne
     */
    public function currentDescription() :HasOne
    {
        return $this->hasOne(PageDescription::class)->where('language_id', currentLanguage()->id);
    }


    public static function withDescription($ids = [])
    {
        $query = self::orderBy('pages.id')->join('page_descriptions as pDesc', 'pDesc.page_id', 'pages.id')
            ->where('pDesc.language_id', currentLanguage()->id);
        if (count($ids)) {
            $query->whereIn('pages.id', $ids);
        }

        $query->select([
            'pages.id',
            'pages.image',
            'pages.images',
            'pages.meta',
            'pDesc.page_title',
            'pDesc.title',
            'pDesc.description'
        ]);

        return count($ids) == 1 ? $query->first() : $query->get();
    }


}
