<?php

namespace App\Http\Controllers\API;

use App\Appointment;
use App\AppointmentTime;
use App\Branch;
use App\Doctor;
use App\Evaluate;
use App\Http\Controllers\Controller;
use App\Http\Requests\Api\AddBookingRequest;
use App\Http\Requests\Web\AddEvaluateRequest;
use App\Http\Resources\BranchResource;
use App\Http\Resources\DoctorResource;
use App\Http\Resources\ReservationResource;
use App\Jobs\SendBookingMessageJob;
use App\Reservation;
use App\Service;
use App\User;
use Illuminate\Foundation\Auth\ThrottlesLogins;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class BookingController extends Controller
{

    use ThrottlesLogins;

    /**
     * The maximum number of attempts to allow.
     *
     * @return int
     */
    protected $maxAttempts = 3;


    /**
     * The number of minutes to throttle for.
     *
     * @return int
     */
    protected $decayMinutes = 1;


    /**
     * The field name throttle trait will use for.
     *
     * @return string
     */
    public function username() : string
    {
        return 'username';
    }


    /**
     * Get booking by booking ID
     * @param $booking_id
     * @return JsonResponse
     */
    public function show($booking_id)
    {
        $booking = Reservation::find($booking_id);

        if (!$booking){
            return response()->json(['message' => __('evaluates.reservation_not_found'), 'data' => null], 400);
        }
        $data = ReservationResource::make($booking);
        return response()->json(['message' => null, 'data' => $data], 200);
    }
    /**
     * Booking step 1
     * Get branches
     * @param $service_id
     * @return JsonResponse
     */
    public function getBranches($service_id)
    {
        $service = Service::withDescription([$service_id])->first();

        if (!$service){
            return response()->json(['message' => __('services.service_not_found'), 'data' => null], 400);
        }

        $branches = Branch::withDescription($service->branch_ids);

        if (!count($branches)){
            return response()->json(['message' => __('branches.no_branches_available'), 'data' => null], 400);
        }

        $data = BranchResource::collection($branches);
        return response()->json(['message' => '', 'data' => $data], 200);
    }

    /**
     * Booking step 2
     * Get doctors
     * @param $service_id
     * @return JsonResponse
     */
    public function getDoctors($service_id)
    {
        $service = Service::withDescription([$service_id])->first();

        if (!$service){
            return response()->json(['message' => __('services.service_not_found'), 'data' => null], 400);
        }

        $doctors = Doctor::join('doctor_descriptions as doctorDesc', 'doctorDesc.doctor_id', 'doctors.id')
            ->where('doctorDesc.language_id', currentLanguage()->id)
            ->whereJsonContains('service_ids', "$service_id")
            ->join('appointments', 'appointments.doctor_id', 'doctors.id')
            ->whereNull('appointments.deleted_at')
            ->where('appointments.service_id', $service_id)
            ->select([
                'doctors.id',
                'doctors.service_ids',
                'doctors.image',
                'doctorDesc.name',
                'doctorDesc.specialist',
            ])->cursor();

        if (!count($doctors)){
            return response()->json(['message' => __('doctors.no_doctors_available'), 'data' => null], 400);
        }

        $data = DoctorResource::collection($doctors);
        return response()->json(['message' => '', 'data' => $data], 200);
    }

    /**
     * Booking step 3
     * Get appointment dates
     * @param Request $request
     * @return JsonResponse
     */
    public function getAppointmentDates(Request $request)
    {

        $service = Service::withDescription([$request->service_id])->first();

        if (!$service){
            return response()->json(['message' => __('services.service_not_found'), 'data' => null], 400);
        }

        if ($request->specific_doctor_way == 1 && $request->doctor_id){

            $doctor = Doctor::withDescription([$request->doctor_id])->first();

            if (!($doctor->id ?? null) || !in_array($request->service_id, ($doctor->service_ids ?? []))){
                return response()->json(['message' => __('doctors.no_doctors_available'), 'data' => null], 400);
            }


            $appointments = Appointment::where('service_id', $request->service_id)
                ->where('doctor_id', $request->doctor_id)
                ->where('appointment_date', '>=', date('Y-m-d'))->get();

            if (!count($appointments)){
                return response()->json(['message' => __('reservations.no_dates_available'), 'data' => null], 400);
            }

            $appointmentDates = $appointments->pluck('appointment_date')->toArray();
            $data = [
                'appointment_date' => array_unique($appointmentDates),
                'doctor_id' => $doctor->id,
                'doctor_name' => $doctor->currentDescription->name,
                'doctor_specialist' => $doctor->specialist,
            ];

            return response()->json(['message' => null, 'data' => $data], 200);
        }

        if ($request->specific_doctor_way == 2 && !$request->doctor_id){

            $doctor = Doctor::whereJsonContains('service_ids', "$request->service_id")
                ->join('appointments', 'appointments.doctor_id', 'doctors.id')
                ->whereNull('appointments.deleted_at')
                ->where('appointments.service_id', $request->service_id)
                ->select(['doctors.id'])
                ->inRandomOrder()->first();

            if (!($doctor->id ?? null)){
                return response()->json(['message' => __('doctors.no_doctors_available'), 'data' => null], 400);
            }


            $appointments = Appointment::where('service_id', $request->service_id)
                ->where('doctor_id', $doctor->id)->get();

            if (!count($appointments)){
                return response()->json(['message' => __('reservations.no_dates_available'), 'data' => null], 400);
            }

            $appointments = $appointments->pluck('appointment_date')->toArray();


            $data = [
                'appointment_date' => $appointments,
                'doctor_id' => $doctor->id,
                'doctor_name' => $doctor->currentDescription->name,
                'doctor_specialist' => $doctor->currentDescription->specialist,
            ];


            return response()->json(['message' => null, 'data' => $data], 200);
        }

    }

    /**
     * Booking step 4
     * Get appointment times
     * @param Request $request
     * @return JsonResponse
     */
    public function getAppointmentTimes(Request $request)
    {
        $appointment = Appointment::where('service_id', $request->service_id)
            ->where('appointment_date', $request->appointment_date)->first();

        if (!($appointment->id ?? null)){
            return response()->json(['message' => __('reservations.no_dates_available'), 'data' => null], 400);
        }

        $appointmentTimes = AppointmentTime::where('appointment_id', $appointment->id)
            ->where('status', 'available')->get()->groupBy('available_time')->toArray();

        if (!count($appointmentTimes)){
            return response()->json(['message' => __('reservations.no_time_available'), 'data' => null], 400);
        }

        $timesAm = [];
        $timesPm = [];
        $timesAmIds = [];
        $timesPmIds = [];

        foreach ($appointmentTimes as $time => $item) {
            $timeAmOrPm = date('h:i A', strtotime($time));
            if (strpos($timeAmOrPm, 'AM')){
                $timesAm[] = $timeAmOrPm;
                $timesAmIds[] = $item[0]['id'];
            }
            if (strpos($timeAmOrPm, 'PM')){
                $timesPm[] = $timeAmOrPm;
                $timesPmIds[] = $item[0]['id'];
            }

        }

        $data = [
            'available_time_am' => $timesAm,
            'available_time_pm' => $timesPm,
            'available_time_am_ids' => $timesAmIds,
            'available_time_pm_ids' => $timesPmIds,
        ];

        return response()->json(['message' => null, 'data' => $data], 200);
    }

    /**
     * Add booking request
     * @param AddBookingRequest $request
     * @param $service_id
     * @return JsonResponse
     */
    public function addBooking(AddBookingRequest $request, $service_id)
    {
        $service = Service::withDescription([$service_id])->first();
        if (!$service){
            return response()->json(['message' => __('services.service_not_found'), 'data' => null], 400);
        }

        $verifyCode = $this->verifyCode($request);

        if(!$verifyCode->original['data']){
            return $verifyCode;
        }

        // Device id (device_id) append to request in SecretTokenMiddleware
        $user = User::where('device_info->device_id', $request->device_id)
            ->where('verify_code', $request->verify_code)
            ->where('phone', $request->phone)
            ->where('email', $request->email)
            ->first();

        if (!$user){
            return $this->checkAttemptCount($request) ?? response()->json(['message' => __('members.invalid_verification_code'), 'data' => null], 400);
        }

        $user->update(['verify_code' => null]);


        $data = $request->all();
        $data['service_id'] = $service->id;
        $data['durations'] = $service->durations;
        $data['doctor_id'] = $request->doctor_id;
        $appointmentTime = AppointmentTime::find($request->appointment_time);
        if (!$appointmentTime){
            return response()->json(['message' => __('reservations.no_time_available'), 'data' => null], 400);
        }

        $appointmentTime = AppointmentTime::find($request->appointment_time);
        if (!$appointmentTime){
            return response()->json(['message' => __('reservations.no_time_available'), 'data' => null], 400);
        }
        $data['appointment_time'] = $appointmentTime->available_time;

        $token = time().'_'.str_random();
        $token = md5($token);
        $data['evaluate_token'] = $token;
        $reservation = Reservation::create($data);

        $appointmentTime->update(['status' => 'not_available']);

        $branch = Branch::find($request->branch_id);
        $doctor = Doctor::find($data['doctor_id']);

        $reservationDetails = [
            'id' => $reservation->id,
            'service_name' => $service->currentDescription->name,
            'service_duration' => $service->durations,
            'branch_name' => $branch->currentDescription->name,
            'branch_address' => $branch->currentDescription->address,
            'branch_phone' => $branch->phone,
            'branch_map' => $branch->map,
            'doctor_name' => $doctor->currentDescription->name,
            'evaluate_token' => $reservation->evaluate_token,
            'appointment_date' => date('F, d, D, Y', strtotime($request->appointment_date)) . ' at ' . date('h:i A', strtotime($request->appointment_time)),
        ];

        //        $phone = explode('0', $user->phone, 2);
        $phone = $user->phone;
        $serviceName = $reservationDetails['service_name'];
        $doctorName = $reservationDetails['doctor_name'];
        $appointmentDate = $reservationDetails['appointment_date'];

        $message = __('reservations.appointment_msg_status_1');
        $message = str_replace_array('@@', [$serviceName, $doctorName, $appointmentDate], $message);
//        try{
//            Sms::send($phone, $message);
//        } catch (Exception $exception){
//            return ['errors' => __('members.sms_error')];
//        }

        try{
            $data = [
                'subject' => __('reservations.reservation_confirmation_msg'),
                'message' => $message,
                'email' => $request->email
            ];
            SendBookingMessageJob::dispatch($data, 'web.bookings.emails.send_email');

        } catch (\Exception $e){}

        return response()->json(['message' => __('reservations.success_message'), 'data' => $reservationDetails], 400);


    }

    public function addEvaluate(AddEvaluateRequest $request, $booking_id)
    {
        $booking = Reservation::where('id', $booking_id)->whereStatus('confirmed')->first();

        if (!$booking){
            return response()->json(['message' => __('evaluates.reservation_not_found'), 'data' => null], 400);
        }

        $token = $request->header('evaluateToken');

        if (!$token){
            return response()->json(['message' => __('evaluates.evaluate_not_allow'), 'data' => null], 400);
        }

        if ($token != $booking->evaluate_token){
            return response()->json(['message' => __('evaluates.evaluate_not_allow'), 'data' => null], 400);
        }

        $isEvaluateBefore = Evaluate::where('reservation_id', $booking->id)->first();


        if (($isEvaluateBefore->id ?? null)){
            return response()->json(['message' => __('evaluates.evaluate_before'), 'data' => null], 400);
        }

        Evaluate::create([
            'name' => $booking->name,
            'reservation_id' => $booking->id,
            'service_id' => $booking->service_id,
            'appointment_date' => $booking->appointment_date,
            'appointment_time' => $booking->appointment_time,
            'rate' => $request->rate,
            'comment' => $request->comment,
        ]);

        return response()->json(['message' => __('evaluates.evaluate_added_successfully'), 'data' => null], 200);

    }

    /**
     * Resend code for register and login
     * @param Request $request
     * @return JsonResponse
     */
    public function sendCode(Request $request)
    {
        $phonePattern = '/^(5){1}([0-9]{8})$/';
        $validation = Validator::make($request->all(), [
            'phone' => ['required','regex:'.$phonePattern],
        ], [], ['phone' => __('reservations.phone')]);
        if ($validation->fails()){
            return response()->json(['message' => $validation->errors()->first(), 'data' => null], 400);
        }


        // Device id (device_id) append to request in SecretTokenMiddleware
        // Device token (device_token) append to request in SecretTokenMiddleware
        // local append to request in SecretTokenMiddleware
        $deviceId = $request->device_id;
        $deviceToken = $request->device_token;
        $local = $request->local;

        $device_info = [
            'device_id' => $deviceId,
            'device_token' => $deviceToken,
            'platform' => null,
        ];

        $user = User::updateOrCreate(
            ['device_info->device_id' => $deviceId],
            [
                'device_info' => $device_info,
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
                'lang' => $local,
                'role' => 'user'
            ]
        );

        if ($this->checkAttemptCount($request)){
            return $this->checkAttemptCount($request);
        }

        $generateVerifyCode = $this->generateVerifyCode($user);

        if ($generateVerifyCode['message']){
            return $this->checkAttemptCount($request) ?? response()->json(['message' => $generateVerifyCode['message'], 'data' => null], 400);
        }

        return response()->json(['message' => __('members.code_send_to_phone'), 'data' => ['code' => $generateVerifyCode['code']]]);
    }

    /**
     * Verify code
     * @param Request $request
     * @return JsonResponse
     */
    public function verifyCode(Request $request)
    {

        $code = $request->verify_code;

        if (mb_strlen($code) < 4){
            return $this->checkAttemptCount($request) ?? response()->json(['message' => __('members.enter_full_code'), 'data' => null], 400);
        }

        // Device id (device_id) append to request in SecretTokenMiddleware
        $user = User::where('device_info->device_id', $request->device_id)
            ->where('verify_code', $code)
            ->where('phone', $request->phone)
            ->where('email', $request->email)
            ->first();


        if (!$user){
            return $this->checkAttemptCount($request) ?? response()->json(['message' => __('members.invalid_verification_code'), 'data' => null], 400);
        } else {
            $user->update([
                'name' => $request->name,
            ]);
        }


        // Check the number of attempts to check the code
        if ($user->code_verify_attempt >= 6 && strtotime($user->last_attempt_date) == strtotime(date('Y-m-d')) ) {
            return $this->checkAttemptCount($request) ?? response()->json(['message' => __('members.verification_tries_exceed'), 'data' => null], 400);
        }

        // Check if date less then today
        if ($user->last_attempt_date && strtotime($user->last_attempt_date) < strtotime(date('Y-m-d'))){
            $user->update(['code_verify_attempt' => 0, 'last_attempt_date' => date('Y-m-d')]);
        }

        // The user entered the wrong code and the number of attempts increased
        if (strtotime($user->last_attempt_date) == strtotime(date('Y-m-d')) && $user->verify_code != $code){
            $user->update(['code_verify_attempt' => $user->code_verify_attempt+1]);
            return $this->checkAttemptCount($request) ?? response()->json(['message' => __('members.invalid_verification_code'), 'data' => null], 400);
        }

//        $user->update(['verify_code' => null]);

        return response()->json(['message' => __('members.correct_code'), 'data' => $code], 200);
    }

    /**
     * Generate verified code
     * @param $user
     * @return array
     */
    private function generateVerifyCode($user)
    {

        /**
         * It's commented for test only and after test
         * delete comment
         */

        if ($user->attempt_send_code >= 6 && strtotime($user->last_attempt_date) == strtotime(date('Y-m-d')) ) {
            return ['message' => __('members.verification_code_exceed'), 'code' => null];
        }

        if ($user->last_attempt_date && strtotime($user->last_attempt_date) < strtotime(date('Y-m-d'))){
            $user->update(['code_verify_attempt' => 0, 'attempt_send_code' => 0,'last_attempt_date' => date('Y-m-d')]);
        }

        $attempt_send_code = $user->attempt_send_code + 1;
        $last_attempt_date = date('Y-m-d');
        $code = rand(1000,9999);

//        $phone = explode('0', $user->phone, 2);
        $phone = $user->phone;

        // This line is temporary and will be deleted later
        $code = in_array($phone, ['581524713', '543605073', '500000000']) ? '1234' : $code;

//        try{
//            $message = __('members.sms_message');
//            $message = str_replace('@@', $code, $message);
//            Sms::send($phone, $message);
//        } catch (Exception $exception){
//            return ['errors' => __('members.sms_error')];
//        }

        /**
         * $data = [
        'user' => $user->name,
        'email' => $user->email,
        'code' => $code,
        ];
        CodeVerificationJob::dispatch($data, 'web.members.emails.verification_email');
         */

        $user->update(
            [
                'verify_code' => $code,
                'attempt_send_code' => $attempt_send_code,
                'last_attempt_date' => $last_attempt_date,
            ]
        );

        return ['message' => null, 'code' => $code];
    }

    /**
     * Check user attempts count
     * @param $request
     * @return JsonResponse
     */
    public function checkAttemptCount($request)
    {
        if (method_exists($this, 'hasTooManyLoginAttempts') &&
            $this->hasTooManyLoginAttempts($request)) {
            $this->fireLockoutEvent($request);
            return $this->sendLockoutResponse($request);
        }

        $this->incrementLoginAttempts($request);

    }

    /**
     * Override parent sendLockoutResponse to handle it
     * to return json response
     * @param Request $request
     * @return JsonResponse
     */
    protected function sendLockoutResponse(Request $request)
    {
        $seconds = $this->limiter()->availableIn(
            $this->throttleKey($request)
        );
        return response()->json(['message' => Lang::get('auth.throttle', [
            'seconds' => $seconds,
            'minutes' => ceil($seconds / 60),
        ]), 'data' => null], 400);
    }



}
